function b64urlEncode(uint8arr) {
  // Convert Uint8Array -> base64url
  let str = "";
  for (let i = 0; i < uint8arr.length; i++) str += String.fromCharCode(uint8arr[i]);
  const b64 = btoa(str);
  return b64.replace(/\+/g, "-").replace(/\//g, "_").replace(/=+$/g, "");
}

function urlBase64ToUint8Array(base64String) {
  // base64url -> Uint8Array
  const padding = "=".repeat((4 - (base64String.length % 4)) % 4);
  const base64 = (base64String + padding).replace(/-/g, "+").replace(/_/g, "/");
  const rawData = atob(base64);
  const outputArray = new Uint8Array(rawData.length);
  for (let i = 0; i < rawData.length; ++i) outputArray[i] = rawData.charCodeAt(i);
  return outputArray;
}

(async () => {
  const statusEl = document.getElementById("status");
  const enableBtn = document.getElementById("enableBtn");
  const iosHelp = document.getElementById("iosHelp");

  const ua = navigator.userAgent || "";
  const isIOS = /iPhone|iPad|iPod/i.test(ua);
  const isStandalone =
    window.matchMedia("(display-mode: standalone)").matches ||
    window.navigator.standalone === true;

  if (isIOS && !isStandalone) iosHelp.hidden = false;

  async function registerSW() {
    if (!("serviceWorker" in navigator)) throw new Error("Service workers not supported.");
    // Ensure sw.js exists at /main/sw.js
    const reg = await navigator.serviceWorker.register(`${window.APP_BASE}sw.js`, { scope: window.APP_BASE });
    await navigator.serviceWorker.ready;
    return reg;
  }

  async function getSubscription() {
    if (!("PushManager" in window)) throw new Error("Push not supported in this browser.");
    if (!window.VAPID_PUBLIC_KEY || window.VAPID_PUBLIC_KEY.length < 40) {
      throw new Error("Missing VAPID public key on page.");
    }

    const reg = await registerSW();
    const existing = await reg.pushManager.getSubscription();
    if (existing) return existing;

    const appServerKey = urlBase64ToUint8Array(window.VAPID_PUBLIC_KEY);

    return reg.pushManager.subscribe({
      userVisibleOnly: true,
      applicationServerKey: appServerKey
    });
  }

  async function saveSubscription(sub) {
    const json = sub.toJSON(); // includes keys in base64 usually
    const p256dh = json?.keys?.p256dh;
    const auth = json?.keys?.auth;

    if (!p256dh || !auth) {
      // fallback: derive keys
      const k = sub.getKey("p256dh");
      const a = sub.getKey("auth");
      if (!k || !a) throw new Error("Subscription keys missing.");
      return saveWithKeys(sub.endpoint, b64urlEncode(new Uint8Array(k)), b64urlEncode(new Uint8Array(a)));
    }

    // Convert to base64url in case browser returns base64
    const p256dhUrl = p256dh.replace(/\+/g, "-").replace(/\//g, "_").replace(/=+$/g, "");
    const authUrl   = auth.replace(/\+/g, "-").replace(/\//g, "_").replace(/=+$/g, "");

    return saveWithKeys(sub.endpoint, p256dhUrl, authUrl);
  }

  async function saveWithKeys(endpoint, p256dh, auth) {
    const payload = {
      org_id: window.ORG.id,
      endpoint,
      p256dh,
      auth,
      user_agent: navigator.userAgent
    };

    const res = await fetch(`${window.APP_BASE}api/push_subscribe.php`, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(payload)
    });

    const data = await res.json().catch(() => ({}));
    if (!res.ok) throw new Error(data?.error || "Failed to save subscription.");
    return data;
  }

  enableBtn.addEventListener("click", async () => {
    enableBtn.disabled = true;
    statusEl.textContent = "Requesting permission…";

    try {
      const perm = await Notification.requestPermission();
      if (perm !== "granted") throw new Error("Permission not granted.");

      statusEl.textContent = "Registering service worker…";
      const sub = await getSubscription();

      statusEl.textContent = "Saving subscription…";
      await saveSubscription(sub);

      statusEl.textContent = "✅ Subscribed! You’ll receive notifications for this organization.";
      enableBtn.textContent = "Enabled";
    } catch (e) {
      statusEl.textContent = "❌ " + (e?.message || "Could not subscribe.");
      enableBtn.disabled = false;
    }
  });
})();
